#include <stdio.h>
#include <stdlib.h>
#include "code.h"
#include "action.h"

unsigned char *mem;
struct code_hdr hdr;
char *prog;
extern int max_xop;
extern char *op_name[];
extern char op_type[];
extern int yyerrcount;
//extern int yydebug;
int nowarn;
int yyopen(char *file);
int yyparse();
void yyclose();
#define BASE	0x8000
#define TOP	0xfff0

long cval[16];
long clast[16];
int interrupt_pending = 0;
int launch_pending = 0;
int launch_recog = 0;
int launch_search = 0;
struct action *input_list = NULL;
int input_offset = 0;
unsigned long now = 0;
unsigned long increment = 2;
unsigned long log_base, log_end;
unsigned long stack_size = 256;
unsigned long poll_addr = 0;
unsigned long extra = 0;
unsigned long trampoline = 0;

void debug(int x) {}

unsigned char 
get_8(FILE *f)
{
	int c = '\n';
	int r;
	
	while (c == '\n' || c == '\r' || c == '!')
		c = fgetc(f);
	if (c >= 'a' && c <= 'f') {
		r = (c-'a'+10)<<4;
	} else
	if (c >= 'A' && c <= 'F') {
		r = (c-'A'+10)<<4;
	} else {
		r = (c-'0')<<4;
	} 
	c = fgetc(f);
	if (c >= 'a' && c <= 'f') {
		r |= (c-'a'+10);
	} else
	if (c >= 'A' && c <= 'F') {
		r |= (c-'A'+10);
	} else {
		r |= (c-'0');
	} 
	return(r);
}

void
load_code(char *name)
{
	FILE *fin;
	long len, addr, tag, sum;
	unsigned char *pnt;
	
	if (name == NULL) {
		fprintf(stderr, "%s: Couldn't open input file\n", prog);
		exit(2);
	}
	fin = fopen(name, "r");
	if (fin == NULL) {
		fprintf(stderr, "%s: Couldn't open input file '%s'\n", prog, name);
		exit(2);
	}
	for (;;) {
		len = get_8(fin);
		addr = (get_8(fin))<<8;
		addr |= get_8(fin);
		tag = get_8(fin);
		if (tag == 1) {
			pnt = (unsigned char *)&hdr;
			if (len > sizeof(hdr))
				len - sizeof(hdr);
		} else {
			pnt = &mem[(BASE+addr)&0xffff];
		}
		while (len--)
			*pnt++ = get_8(fin);
		sum = get_8(fin);
		if (tag)
			break;
	}
	
	fclose(fin);
}

#define	POP(x)	{x = mem[sp++];x |= (mem[sp++]<<8);}
#define	POP4(x)	{x = mem[sp++];x |= (mem[sp++]<<8);x |= (mem[sp++]<<16);x |= (mem[sp++]<<24);}
#define	PUSH(x)	{mem[--sp] = x>>8;mem[--sp] = x;}
#define	PUSH4(x){mem[--sp] = x>>24;mem[--sp] = x>>16;mem[--sp] = x>>8;mem[--sp] = x;}
#define	N2	(mem[pc]|(mem[pc+1]<<8))
#define	N4	(mem[pc]|(mem[pc+1]<<8)|(mem[pc+2]<<16)|(mem[pc+3]<<24))
#define	XN2	(mem[pc+2]|(mem[pc+3]<<8))
#define	XN4	(mem[pc+2]|(mem[pc+3]<<8)|(mem[pc+4]<<16)|(mem[pc+5]<<24))

unsigned long	poll_time, current_time, launch_time, gtime, log_full;

void 
do_fire(unsigned short tos)
{
	printf("fire(%d)\n",tos);
}

void 
do_safe(unsigned short tos)
{
	printf("safe(%d)\n",tos);
}

void 
do_arm(unsigned short tos)
{
	printf("arm(%d)\n",tos);
}

void 
do_beep(unsigned short tos)
{
	printf("beep(%d)\n",tos);
}

void
do_set(unsigned short tos, unsigned short ntos)
{
	printf("set(%d, %d)\n",tos, ntos);
}

void 
do_eeload(unsigned short tos, unsigned short ntos, unsigned short addr)
{
	printf("eeload(%d, %d, %d)\n",tos, ntos, addr);
}

void 
do_eesave(unsigned short tos, unsigned short ntos, unsigned short addr)
{
	printf("eesave(%d, %d, %d)\n",tos, ntos, addr);
}


unsigned short 
do_output()
{
	return(1);
}

struct action *alist = NULL;
struct action *alast = NULL;
struct action *this_action = NULL;

struct action *new_action()
{
	struct action *ap;
	int i;

	ap = (struct action *)malloc(sizeof(*ap));
	ap->next = 0;
	for (i = 0; i < 16; i++) {
		ap->type[i] = 0;
		ap->chan[i] = 0;
	}
	ap->interrupt = 0;
	ap->input = NULL;
	return(ap);
}

void
set_timescale(unsigned long t)
{
	if (t == 0)
		t = 1;
	increment = t;
}

void
save_action(unsigned long t)
{
	this_action->time = t;
	if (alast) {
		if (alast->time >= t) {
			fprintf(stderr, "%s: stimulus time must increase in order\n", prog);
			exit(2);
		}
		alast->next = this_action;
	} else {
		if (t != 0) {
			fprintf(stderr, "%s: stimulus time must begin at 0\n", prog);
			exit(2);
		}
		alist = this_action;
	}
	alast = this_action;
	this_action = new_action();
}
			

void
load_stim()
{
	struct action *ap;
	this_action = new_action();
	if(yyparse() || yyerrcount > 0) 
		exit(2);
	yyclose();
	//for (ap=alist;ap;ap=ap->next) {
	//	printf("ap->time=%d, ap->val=%d/%d %d/%d\n", ap->time, ap->chan[0], ap->type[0], ap->chan[1], ap->type[1]);
	//}
}

unsigned short 
do_interrupt(unsigned short tos)
{
	if (interrupt_pending&(1<<tos)) {
		interrupt_pending &= ~(1<<tos);
		return(1);
	}
	return(0);
}

unsigned short 
do_launch()
{
	if (launch_recog) {
		launch_recog = 0;
		launch_search = 0;
		return(1);
	}
	launch_search = 1;
	return(0);
}

unsigned short 
do_get(unsigned short tos)
{
	if (tos >= 16)
		return(0);
	return(cval[tos]);
}

unsigned short 
do_input()
{
	char c;

	if (input_list) {
		for (;;) {
			c = input_list->input[input_offset];
			if (c != 0) {
				input_offset++;
				break;
			}
			c = 0;
			for (;;) {
				input_list = input_list->next;
				if (input_list == NULL)
					break;
				if (input_list->time > current_time) {
					input_list = NULL;
					break;
				}
				if (input_list->input) {
					input_offset = 0;
					break;
				}
			}
			if (input_list == NULL)
				break;
		}
	} else {
		c = 0;
	}
	return(c);
}



int
main(int argc, char **argv)
{
	long tos, ntos;
	unsigned short addr;
	unsigned short pc;
	unsigned short gp;
	unsigned short sp;
	unsigned short fp;
	unsigned short strp;
	unsigned short code_size;
	unsigned short global_size;
	unsigned short string_size;
	struct action *ap;
	long quit, i, op, dbg=0;
	char *code_file, *cp;
	unsigned char *up;
	int stim = 0;
	
	mem = (unsigned char *)malloc(64*1024);
	hdr.magic = 0xaa;
	code_file = NULL;
	prog = argv[0];
	for (i = 1;i < argc; i++) {
		if (argv[i][0] == '-') {
			cp = &argv[i][1];
			for (quit = 0;!quit;) {
				switch (*cp++) {
				case 0:
					quit = 1;
					break;
				
				case 'w':
					nowarn = 1 - nowarn;
					break;

				case 'D':
					//yydebug = 1;
					break;

				case 'd':
					dbg = 1 - dbg;
					break;

				case 's':
					if (stim) {
						fprintf(stderr, "%s: multiple stimulus files\n", prog);
                                        	exit(2);
					}
					if (*cp == 0) {
						i++;
						if (i == argc) {
							fprintf(stderr, "%s: missing argument for -s\n", prog);
                                        		exit(2);
						}
						cp = argv[i];
					}
					if (yyopen(cp)) {
						fprintf(stderr, "%s: unable to open stimulus file '%s'\n", prog, cp);
                                       		exit(2);
					}
					stim = 1;
					cp="";
					break;
					
				default:
					fprintf(stderr, "%s: unknown flag -%c\n", prog, cp[-1]);
					exit(2);
				}
			}
		} else {
			if (code_file) {
				fprintf(stderr, "%s: multiple input files '%s' and '%s'\n", prog, code_file, argv[i]);
				exit(2);
			}
			code_file = argv[i];
		}
	}
	if (stim) {
		load_stim();
	}
	load_code(code_file);
	if (hdr.magic != MAGIC) {
		fprintf(stderr, "Invalid magic number on input file = 0x%x\n", hdr.magic);
		exit(2);
	}
	code_size = hdr.code_size[0]|(hdr.code_size[1]<<8);
	global_size = hdr.global_size[0]|(hdr.global_size[1]<<8);
	string_size = hdr.string_size[0]|(hdr.string_size[1]<<8);
	stack_size = hdr.stack_size[0]|(hdr.stack_size[1]<<8);
	fp = 0;
	pc = BASE;
	strp = BASE+code_size;
	gp = BASE+code_size+string_size;
	for (i = gp; i < (gp+global_size); i++)
		mem[i] = 0;
	sp = TOP;
	log_base = gp+global_size;
	log_end = sp-stack_size;
	tos = 0;
	for(quit = 0;!quit;) {
		if (alist) {
			if ((current_time == 0 && now == 0) || (alist->next && current_time >= alist->next->time)) {
				if (now != 0 || current_time != 0)
					alist = alist->next;
				interrupt_pending |= alist->interrupt;
				if (alist->input && input_list == NULL) {
					input_list = alist;
					input_offset = 0;
				}
				for (i = 0; i < 16; i++) 
				if (alist->type[i]) {
					clast[i] = cval[i] = alist->chan[i];
				} else {
					clast[i] = cval[i];
				}
			}
			if (ap = alist->next) {
				for (i = 0; i < 16; i++) 
				if (ap->type[i] == 2) {
					cval[i] = clast[i]+(long)((float)(current_time-alist->time)*(ap->chan[i]-clast[i])/(float)(ap->time-alist->time));
				}
			}
		}
		op = mem[pc++];
		if (dbg) {
			printf("%3d.%03d @", current_time/1000, current_time%1000);
			if (op <= max_xop) {
				switch (op_type[op]) {
				case 0:	printf("%04x: %-20s ",pc-1,op_name[op]);
					break;
				case 1:	printf("%04x: %-15s %04x " ,pc-1,op_name[op], (N2+pc)&0xffff);
					break;
				case 2:	printf("%04x: %-15s %04x " ,pc-1,op_name[op], N2);
					break;
				case 3:	printf("%04x: %-15s %04x ",pc-1,op_name[op], N2);
					break;
				case 4:	printf("%04x: %-15s %04x ",pc-1,op_name[op], N2);
					break;
				case 5:	printf("%04x: %-15s %04x ",pc-1,op_name[op], N4);
					break;
				}
			} else {
				printf("%04x: BAD! - %02x ",pc-1,op_name[op], tos);
			}
			printf("tos=%04x stack[%04x] = %04x %04x\n", tos, sp, mem[sp]|(mem[sp+1]<<8), mem[sp+2]|(mem[sp+3]<<8));
		}
		switch (op){
		case P_HALT:		// halt
			quit = 1;
			break;
		case P_J_NE:		// jump if TOS ne 0
			if (tos) {
				pc += N2;
			} else {
				pc += 2;
			}
			break;
		case P_J_EQ:		// jump if TOS eq 0
			if (!tos) {
				pc += N2;
			} else {
				pc += 2;
			}
			break;
		case P_JEQV:		// jump if TOS eq V
			if (XN2 == (short)tos) {
				pc += N2;
			} else {
				pc += 4;
			}
			break;
		case P_JEQV_L:		// jump if TOS eq V
			if (XN4 == tos) {
				pc += N2;
			} else {
				pc += 6;
			}
			break;
		case P_JNEV:		// jump if TOS eq V
			if (XN2 != (short)tos) {
				pc += N2;
			} else {
				pc += 4;
			}
			break;
		case P_JNEV_L:		// jump if TOS eq V
			if (XN4 != tos) {
				pc += N2;
			} else {
				pc += 6;
			}
			break;
		
		case P_GEU:		// tos is replaced with truth value for tos-1 >= tos (unsigned)
			POP(ntos);
			tos = (unsigned short)ntos >= (unsigned short)tos;
			break;
		case P_GTU:		// tos is replaced with truth value for tos-1 > tos (unsigned)
			POP(ntos);
			tos = (unsigned short)ntos > (unsigned short)tos;
			break;
		case P_LEU:		// tos is replaced with truth value for tos-1 <= tos (unsigned)
			POP(ntos);
			tos = (unsigned short)ntos <= (unsigned short)tos;
			break;
		case P_LTU:		// tos is replaced with truth value for tos-1 < tos (unsigned)
			POP(ntos);
			tos = (unsigned short)ntos < (unsigned short)tos;
			break;
		case P_GE:		// tos is replaced with truth value for tos-1 >= tos
			POP(ntos);
			tos = (short)ntos >= (short)tos;
			break;
		case P_GT:		// tos is replaced with truth value for tos-1 > tos
			POP(ntos);
			tos = (short)ntos > (short)tos;
			break;
		case P_LE:		// tos is replaced with truth value for tos-1 <= tos
			POP(ntos);
			tos = (short)ntos <= (short)tos;
			break;
		case P_LT:		// tos is replaced with truth value for tos-1 < tos
			POP(ntos);
			tos = (short)ntos < (short)tos;
			break;
		case P_NE:		// tos is replaced with truth value for tos-1 != tos
			POP(ntos);
			tos = (short)ntos != (short)tos;
			break;
		case P_EQ:		// tos is replaced with truth value for tos-1 == tos
			POP(ntos);
			tos = (short)ntos == (short)tos;
			break;
		case P_EQ_C:		// tos is replaced with truth value for tos-1 == tos
			tos = (short)N2 == (short)tos;
			pc += 2;
			break;
		case P_EQ_L_C:		// tos is replaced with truth value for tos-1 == tos
			tos = N4 == tos;
			pc += 4;
			break;
		case P_NOT:		// tos is replaced with truth value for tos == 0
			tos = !(short)tos;
			break;
		case P_SUB:		// tos is replaced with value for tos-1 - tos
			POP(ntos);
			tos = (short)ntos - (short)tos;
			break;
		case P_ADD:		// tos is replaced with value for tos-1 + tos
			POP(ntos);
			tos = (short)ntos + (short)tos;
			break;
		case P_ADD_C:		// tos is replaced with value for tos-1 + tos
			tos = (short)N2 + (short)tos;
			pc += 2;
			break;
		case P_ADD_L_C:		// tos is replaced with value for tos-1 + tos
			tos = N4 + tos;
			pc += 4;
			break;
		case P_MULU:		// tos is replaced with value for tos-1 * tos
			POP(ntos);
			tos = (unsigned short)ntos * (unsigned short)tos;
			break;
		case P_MUL:		// tos is replaced with value for tos-1 * tos
			POP(ntos);
			tos = (short)ntos * (short)tos;
			break;
		case P_DIVU:		// tos is replaced with value for tos-1 / tos
			POP(ntos);
			tos = ((unsigned short)tos==0?0:(unsigned short)ntos / (unsigned short)tos); // actually on div by 0 TOS is undefined
			break;
		case P_DIV:		// tos is replaced with value for tos-1 / tos
			POP(ntos);
			tos = ((short)tos==0?0:(short)ntos / (short)tos); // actually on div by 0 TOS is undefined
			break;
		case P_MOD:		// tos is replaced with value for tos-1 % tos
			POP(ntos);
			tos = ((short)tos==0?0:(short)ntos % (short)tos); // actually on mod by 0 TOS is undefined
			break;
		case P_OR:		// tos is replaced with value for tos-1 | tos
			POP(ntos);
			tos = (short)ntos | (short)tos;
			break;
		case P_XOR:		// tos is replaced with value for tos-1 ^ tos
			POP(ntos);
			tos = (short)ntos ^ (short)tos;
			break;
		case P_AND:		// tos is replaced with value for tos-1 & tos
			POP(ntos);
			tos = (short)ntos & (short)tos;
			break;
		case P_SHL:		// tos is replaced with value for tos-1 << tos
			POP(ntos);
			tos = (short)ntos << (short)tos;
			break;
		case P_INDEX2:		// tos is replaced with value for tos-1 << tos
			tos = (short)tos << 1;
			break;
		case P_INDEX4:		// tos is replaced with value for tos-1 << tos
			tos = (short)tos << 2;
			break;
		case P_SHR:		// tos is replaced with value for tos-1 >> tos
			POP(ntos);
			tos = (short)ntos >> (short)tos;
			break;
		case P_JMP:		// jump unconditionally
			pc += N2;
			break;
		case P_COMP:		// tos is replaced with ~tos 1's complement)
			tos = ~(short)tos;
			break;
		case P_CALL:		// call subroutine
			tos = pc+2;
			pc += N2;
			break;
		case P_CALLI:		// call-indirect subroutine (from TOS - TOS is popped)
			ntos = tos;
			tos = pc;
			pc = ntos;
			break;
		case P_COPYN:		// N/2+1 words are popped from TOS, old TOS is pushed back on
			sp += N2;
			pc += 2;
			break;
		case P_CUTN:		// N/2 words are popped from TOS
			sp += N2;
		//	POP(tos);
			pc += 2;
			break;
		case P_CONST:		// push a constant value
			tos = N2;
			pc += 2;
			break;
		case P_ADDR_L:		// push the address of a local value  (2 byte offset)
			tos = N2+fp;
			pc += 2;
			break;
		case P_ADDR_G:		// push the address of a global value  (2 byte offset)
			tos = N2+gp;
			pc += 2;
			break;
		case P_BYTE_L:		// push a local byte value  (2 byte offset)
			tos = mem[N2+fp];
			if (tos&0x80)
				tos |= 0xff00;
			pc += 2;
			break;
		case P_UBYTE_L:		// push a local byte value  (2 byte offset)
			tos = mem[N2+fp];
			pc += 2;
			break;
		case P_UBYTE_G:		// push a global byte value  (2 byte offset)
			tos = mem[N2+gp];
			pc += 2;
			break;
		case P_BYTE_G:		// push a global byte value  (2 byte offset)
			tos = mem[N2+gp];
			if (tos&0x80)
				tos |= 0xff00;
			pc += 2;
			break;
		case P_WORD_L:		// push a local word value  (2 byte offset)
			tos = mem[N2+fp]|(mem[N2+fp+1]<<8);
			pc += 2;
			break;
		case P_WORD_G:		// push a global word value  (2 byte offset)
			tos = mem[N2+gp]|(mem[N2+gp+1]<<8);
			pc += 2;
			break;
		case P_SBYTE_L:		// store a local byte value  (2 byte offset)
			mem[N2+fp] = tos;
			pc += 2;
			break;
		case P_SBYTE_G:		// store a global byte value  (2 byte offset)
			mem[N2+gp] = tos;
			pc += 2;
			break;
		case P_SWORD_L:		// store a local word value  (2 byte offset)
			mem[N2+fp] = tos;
			mem[N2+fp+1] = tos>>8;
			pc += 2;
			break;
		case P_SWORD_G:		// store a global word value  (2 byte offset)
			mem[N2+gp] = tos;
			mem[N2+gp+1] = tos>>8;
			pc += 2;
			break;
		case P_LOAD_UB:		// tos is replaced by byte from indirection of old tos value
			tos = (tos+N2)&0xffff;
			tos = mem[tos];
			if (tos&0x80)
				tos |= 0xff00;
			pc += 2;
			break;
		case P_LOAD_B:		// tos is replaced by byte from indirection of old tos value
			tos = (tos+N2)&0xffff;
			tos = mem[tos];
			pc += 2;
			break;
		case P_LOAD_W:		// tos is replaced by word from indirection of old tos value
			tos = (tos+N2)&0xffff;
			tos = mem[tos]|(mem[tos+1]<<8);
			pc += 2;
			break;
		case P_DUP:		// duplicate top of stack
			PUSH(tos);
			break;
		case P_STORE_B:		// *tos = tos-1 (byte) tos/tos-1 are discarded
			POP(ntos);
			tos = (tos+N2)&0xffff;
			mem[tos] = ntos;
			pc += 2;
			break;
		case P_STORE_W:		// *tos = tos-1 (word) tos/tos-1 are discarded
			POP(ntos);
			tos = (tos+N2)&0xffff;
			mem[tos] = ntos;
			mem[tos+1] = ntos>>8;
			pc += 2;
			break;
		case P_ADDR_P:		// push the address of a code address (relative) (2 byte offset)
			tos = pc + N2;
			pc += 2;
			break;
		case P_ZERO:		// pushes 0
			tos = 0;
			break;
		case P_ONE:		// pushes 1
			tos = 1;
			break;
		case P_ENTER:		// allocates space for local variables (subtracts sp by 2 byte offset)
			PUSH(tos);
			if (N2 != 0xffff) {
				PUSH(fp);
				sp -= N2;
				fp = sp;
			}
			pc +=2;
			break;
		case P_RET:		// recovers space for local variables (subtracts sp by 2 byte offset)
			if (N2 != 0xffff) {
				sp += N2;
				POP(fp);
			}
			POP(pc);
			break;
		case P_ADDR_S:		// push the address of a string (2 byte offset)
			tos = strp+N2;
			pc += 2;
			break;
		case P_POLL:		// call into environment for busy work (logging, lauch detect etc)
			poll_time = current_time;
			if (poll_addr) {
				tos = pc;
				pc = poll_addr;
			}
			break;
		case P_INTERRUPT:	// push true if interrupt N has occured (and clear the flag)(2 byte number)
			tos = do_interrupt(N2);	
			pc += 2;
			break;
		case P_SAVE_TIME:	// saves current time
			gtime = current_time;
			break;
		case P_CMP_TIME:	// pushes true if current time is N > than the last saved one (2 byte constant)
			tos = (current_time >= (gtime+N4));
			pc += 4;
			break;
		case P_INPUT:		// pushes false if no input char available, and the chare and true it it is
			i = do_input();
			if (i == 0) {
				tos = 0;
			} else {
				tos = i;
			}
			break;
		case P_OUTPUT:		// pushes truth value - true if we can send a character without stalling
			tos = do_output();	
			break;
		case P_LAUNCH:		// pushes truth value - true if we have launched
			tos = do_launch();	
			break;
		case P_FOUND_LAUNCH:
			
			launch_time = poll_time;
			launch_recog = 1;
			break;
		case P_LOG_FULL:	// pushes truth value - true if the log is full
			tos = log_full;	
			break;
		case P_ARM:		// pops tos and passes it to the arming mechanism
			do_arm(tos);
			if (extra) {
				PUSH(pc);
				PUSH(0);
				PUSH(tos);
				PUSH(0x44);
				tos = trampoline;
				pc = extra;
			}
			break;
		case P_SAFE:		// pops tos and passes it to the safing mechanism
			do_safe(tos);
			if (extra) {
				PUSH(pc);
				PUSH(0);
				PUSH(tos);
				PUSH(0x45);
				tos = trampoline;
				pc = extra;
			}
			break;
		case P_FIRE:		// pops tos and uses it to activate a pyro channel
			do_fire(tos);
			if (extra) {
				PUSH(pc);
				PUSH(0);
				PUSH(tos);
				PUSH(0x43);
				tos = trampoline;
				pc = extra;
			}
			break;
		case P_TIME:		// pushes absolute time
			tos = poll_time;
			break;
		case P_LTIME:		// pushes time since launch event
			tos = poll_time-launch_time;
			break;
		case P_STIME:		// pushes time since launch event
			tos = gtime;
			break;
		case P_RTIME:		// pushes time since launch event
			tos = current_time;
			break;
		case P_LOG_BASE:	// pushes 
			tos = log_base;
			break;
		case P_LOG_END:	// pushes 
			tos = log_end;
			break;
		case P_SET_POLL:	// pushes 
			poll_addr = pc+N2;
			pc+= 2;
			break;
		case P_SET_EXTRA:	// pushes 
			extra = pc+N2;
			pc+= 2;
			break;
		case P_SET_TRAMPOLINE:	// pushes 
			trampoline = pc+N2;
			pc+= 2;
			break;
		case P_SET_LOG_FULL:	// pushes 
			log_full = tos;
			break;
		case P_BEEP:		// pops TOS value and uses it to control the beeper (if any)
			do_beep(tos);
			break;
		case P_PSTR:		// pops TOS value and uses it as the address of a string to print
			tos = tos&0xffff;
			for (;;tos++) {
				if (mem[tos] == 0)
					break;
				if (mem[tos] == 0xa) {
					printf("\n");
				} else {
					printf("%c", mem[tos]);
				}
			}
			break;
		case P_PHEX:		// pops TOS value and uses it as a hex value to print
			printf("%04x", tos&0xffff);
			break;
		case P_PVAL:		// pops TOS value and uses it as a decimal value to print
			printf("%d", tos);
			break;
		case P_PVALU:		// pops TOS value and uses it as a decimal value to print
			printf("%u", tos);
			break;
		case P_PHEXL:		// pops TOS value and uses it as a hex value to print
			printf("%08x", tos);
			break;
		case P_PCHR:		// pops TOS value and uses it as a character value to print
			if (tos == 0xa) {
				printf("\n");
			} else {
				printf("%c", tos);
			}
			break;
		case P_EESAVE:		// pops 3 TOS values saves data to eeprom tos is eeprom address, tos-1 main mem address tos-2 is count
			POP(addr);
			POP(ntos);
			do_eesave(tos, ntos, addr);
			break;
		case P_EELOAD:		// pops 3 TOS values loads data from eeprom tos is main mem address, tos-1 eeprom address tos-2 is count
			POP(addr);
			POP(ntos);
			do_eeload(tos, ntos, addr);
			break;
		case P_GET:		// get value from external source (tos gives value and is replaced with old one)
			tos = do_get(tos);
			break;
		case P_SET:		// set value to external source (tos is the value tos-1 points to the source, both are removed)
			POP(ntos);
			do_set(tos, ntos);
			if (extra) {
				PUSH(pc);
				PUSH(ntos);
				PUSH(tos);
				PUSH(0x46);
				tos = trampoline;
				pc = extra;
			}
			break;
		case P_POP:		// discards the tos
			POP(tos);
			break;
		case P_POP_L:		// discards the tos
			POP4(tos);
			break;
		case P_GEU_L:
			POP4(ntos);
			tos = (unsigned long)ntos >= (unsigned long)tos;
			break;
		case P_LEU_L:
			POP4(ntos);
			tos = (unsigned long)ntos <= (unsigned long)tos;
			break;
		case P_LTU_L:
			POP4(ntos);
			tos = (unsigned long)ntos < (unsigned long)tos;
			break;
		case P_GTU_L:
			POP4(ntos);
			tos = (unsigned long)ntos > (unsigned long)tos;
			break;
		case P_GE_L:
			POP4(ntos);
			tos = ntos >= tos;
			break;
		case P_LE_L:
			POP4(ntos);
			tos = ntos <= tos;
			break;
		case P_LT_L:
			POP4(ntos);
			tos = ntos < tos;
			break;
		case P_GT_L:
			POP4(ntos);
			tos = ntos > tos;
			break;
		case P_NE_L:
			POP4(ntos);
			tos = ntos != tos;
			break;
		case P_EQ_L:
			POP4(ntos);
			tos = ntos == tos;
			break;
		case P_NOT_L:		// tos is replaced with truth value for tos == 0
			tos = !tos;
			break;
		case P_SUB_L:		// tos is replaced with value for tos-1 - tos
			POP4(ntos);
			tos = ntos - tos;
			break;
		case P_ADD_L:		// tos is replaced with value for tos-1 + tos
			POP4(ntos);
			tos = ntos + tos;
			break;
		case P_MULU_L:		// tos is replaced with value for tos-1 * tos
			POP4(ntos);
			tos = ntos * tos;
			break;
		case P_MUL_L:		// tos is replaced with value for tos-1 * tos
			POP4(ntos);
			tos = (unsigned long)ntos * (unsigned long)tos;
			break;
		case P_DIVU_L:		// tos is replaced with value for tos-1 / tos
			POP4(ntos);
			tos = ((unsigned long) tos==0?0:(unsigned long) ntos / (unsigned long) tos); // actually on div by 0 TOS is undefined
			break;
		case P_DIV_L:		// tos is replaced with value for tos-1 / tos
			POP4(ntos);
			tos = (tos==0?0:ntos / tos); // actually on div by 0 TOS is undefined
			break;
		case P_MOD_L:		// tos is replaced with value for tos-1 % tos
			POP4(ntos);
			tos = (tos==0?0:ntos % tos); // actually on mod by 0 TOS is undefined
			break;
		case P_OR_L:		// tos is replaced with value for tos-1 | tos
			POP4(ntos);
			tos = ntos | tos;
			break;
		case P_XOR_L:		// tos is replaced with value for tos-1 ^ tos
			POP4(ntos);
			tos = ntos ^ tos;
			break;
		case P_AND_L:		// tos is replaced with value for tos-1 & tos
			POP4(ntos);
			tos = ntos & tos;
			break;
		case P_SHL_L:		// tos is replaced with value for tos-1 << tos
			POP4(ntos);
			tos = ntos << tos;
			break;
		case P_SHR_L:		// tos is replaced with value for tos-1 >> tos
			POP4(ntos);
			tos = ntos >> tos;
			break;
		case P_COMP_L:		// tos is replaced with ~tos 1's complement)
			tos = ~tos;
			break;
		case P_CONST_L:		// push a constant value
			tos = N4;
			pc += 4;
			break;
		case P_SLONG_L:		// store a local byte value  (2 byte offset)
			mem[N2+fp] = tos;
			mem[N2+fp+1] = tos>>8;
			mem[N2+fp+2] = tos>>16;
			mem[N2+fp+3] = tos>>24;
			pc += 2;
			break;
		case P_SLONG_G:		// store a local byte value  (2 byte offset)
			mem[N2+gp] = tos;
			mem[N2+gp+1] = tos>>8;
			mem[N2+gp+2] = tos>>16;
			mem[N2+gp+3] = tos>>24;
			pc += 2;
			break;
		case P_LONG_L:		// push a local byte value  (2 byte offset)
			tos = mem[N2+fp]|(mem[N2+fp+1]<<8)|(mem[N2+fp+2]<<16)|(mem[N2+fp+3]<<24);
			pc += 2;
			break;
		case P_LONG_G:		// push a local byte value  (2 byte offset)
			tos = mem[N2+gp]|(mem[N2+gp+1]<<8)|(mem[N2+gp+2]<<16)|(mem[N2+gp+3]<<24);
			pc += 2;
			break;
		case P_LOAD_L:		// tos is replaced by word from indirection of old tos value
			tos = (tos+N2)&0xffff;
			tos = mem[tos]|(mem[tos+1]<<8)|(mem[tos+2]<<16)|(mem[tos+3]<<24);
			pc += 2;
			break;
		case P_DUP_L:		// duplicate top of stack
			PUSH4(tos);
			break;
		case P_STORE_L:		// *tos = tos-1 (byte) tos/tos-1 are discarded
			POP4(ntos);
			tos = (tos+N2)&0xffff;
			mem[tos] = ntos;
			mem[tos+1] = ntos>>8;
			mem[tos+2] = ntos>>16;
			mem[tos+3] = ntos>>24;
			pc += 2;
			break;
		case P_ZERO_L:		// zero top of stack
			tos = 0;
			break;
		case P_ONE_L:		// zero top of stack
			tos = 1;
			break;
		case P_WIDENU:
			tos = tos&0xffff;
			break;
		case P_WIDEN:
			tos = (tos&0x8000 ? tos|0xffff0000:tos&0xffff);
			break;
		case P_LSHORTEN:			// preserve long logical value
			tos = (tos|(tos>>16))|0xffff;
			break;
		case P_SHORTEN:
			tos = tos&0xffff;
			break;
		default:
			fprintf(stderr, "Invalid opcode 0x%x @ 0x%04x\n",op, pc-1);
			exit(2);
		}
		now ++;
		if (now >= increment) {
			current_time++;
			now = 0;
		}
	}
	return(1);
}

